<?php
/**
 * Plugin Name: Unific
 * Plugin URI: http://www.unific.com/
 * Description: Unific integrates Woocommerce with Marketing Platform like HubSpot.
 * Version: 1.0.9
 * Author: Unific
 * Author URI: http://www.unific.com/
 * Text Domain: unific
 * Domain Path: /i18n/languages/
 * Requires at least: 4.1
 * Tested up to: 6.4.2
 * Woocommerce version at least: 2.4
 *
 * @package Unific
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/*
*
* Check if WooCommerce is active
*
*/
if ( in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ), true ) ) {

	/* ***************** Newly added for WC abandoned cart  *************************** */

	register_activation_hook( __FILE__, 'unific_activate' );
	// fire de-activate event after de-activate the plugin.
	register_deactivation_hook( __FILE__, 'unific_deactivate' );
	// fire unistall event after deletion of plugin.
	register_uninstall_hook( __FILE__, 'unific_uninstall' );

	// Put plugin activation code here.
	if ( ! function_exists( 'unific_activate' ) ) {
		/**
		 * Activation function for the Unific plugin.
		 */
		function unific_activate() {
			global $wpdb;
			$wcap_collate = '';
			if ( $wpdb->has_cap( 'collation' ) ) {
				$wcap_collate = $wpdb->get_charset_collate();
			}

			$table_name = $wpdb->prefix . 'usermeta_unific';
			$index1     = "ix_{$table_name}_entityid_entitytype";
			$index2     = "ix_{$table_name}_entityid_entitytype_metakey";
			$sql        = "CREATE TABLE IF NOT EXISTS {$table_name} (

				`unific_meta_id` int(11) NOT NULL AUTO_INCREMENT,

				`entity_id` varchar(20) COLLATE utf8_unicode_ci NOT NULL,

				`entity_type` varchar(45) COLLATE utf8_unicode_ci NOT NULL,

				`created` datetime NULL,

				`modified` datetime NULL,

				`meta_key` varchar(100) COLLATE utf8_unicode_ci NOT NULL,

				`meta_value` mediumtext COLLATE utf8_unicode_ci NOT NULL,

				PRIMARY KEY (`unific_meta_id`)

					) {$wcap_collate} AUTO_INCREMENT=1 ";

			require_once ABSPATH . 'wp-admin/includes/upgrade.php';
			dbDelta( $sql );

			// check and create index1.
			// phpcs:ignore
			$sql_check_index1_result = $wpdb->get_results( $wpdb->prepare( 'SELECT * FROM INFORMATION_SCHEMA.STATISTICS WHERE table_schema=DATABASE() AND table_name = %s AND index_name = %s', $table_name, $index1 ) );
			if ( count( $sql_check_index1_result ) === 0 ) {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery
				$wpdb->get_results( "CREATE INDEX {$index1} ON {$table_name} (`entity_id`,`entity_type`)" ); // phpcs:ignore
			}

			// check and create index2.
			// phpcs:ignore
			$sql_check_index2_result = $wpdb->get_results( $wpdb->prepare( 'SELECT * FROM INFORMATION_SCHEMA.STATISTICS WHERE table_schema=DATABASE() AND table_name = %s AND index_name = %s', $table_name, $index2 ) );
			if ( count( $sql_check_index2_result ) === 0 ) {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery
				$wpdb->get_results( "CREATE INDEX {$index2} ON {$table_name} (`entity_id`,`entity_type`,`meta_key`)" ); // phpcs:ignore
			}
		}
	}

	// Put plugin de-activate code here.
	if ( ! function_exists( 'unific_deactivate' ) ) {
		/**
		 * De-Activation function for the Unific plugin.
		 */
		function unific_deactivate() {
		}
	}

	/* ***************************** End here  ************************************ */

	if ( ! function_exists( 'unific_save_extra_checkout_fields' ) ) {
		/**
		 * Unific_save_extra_checkout_fields and $order_id.
		 *
		 * @param int $order_id The ID of the order being processed.
		 */
		function unific_save_extra_checkout_fields( $order_id ) {
			// don't forget appropriate sanitization if you are using a different field type.
			$unific_affiliate_id = ! empty( $_COOKIE['unific_affiliate_id'] ) ? sanitize_text_field( wp_unslash( $_COOKIE['unific_affiliate_id'] ) ) : null;
			setcookie( 'unific_affiliate_id', '', time() - ( 60 * 60 ), '/' );
			if ( ! empty( $unific_affiliate_id ) ) {
				update_post_meta( $order_id, 'affiliate', $unific_affiliate_id );
			}

			// don't forget appropriate sanitization if you are using a different field type.
			$hubspotutk = ! empty( $_COOKIE['hubspotutk'] ) ? sanitize_text_field( wp_unslash( $_COOKIE['hubspotutk'] ) ) : null;
			if ( ! empty( $hubspotutk ) ) {
				update_post_meta( $order_id, 'hubspotutk', $hubspotutk );
			}

			/*
			 * Newly added for WC abandoned cart
			 */
			// don't forget appropriate sanitization if you are using a different field type.
			$unific_cart_hash = ! empty( $_COOKIE['unific_cart_hash'] ) ? sanitize_text_field( wp_unslash( $_COOKIE['unific_cart_hash'] ) ) : null;
			setcookie( 'unific_cart_hash', '', time() - ( 60 * 60 ), '/' );
			if ( ! empty( $unific_cart_hash ) ) {
				update_post_meta( $order_id, 'unific_cart_hash', $unific_cart_hash );
			}
			/* ***************************** End here  ************************************ */

			// update order and user meta.
			$unific_newsletter_subscription = ! empty( $_COOKIE['unific_newsletter_subscription'] ) ? sanitize_text_field( wp_unslash( $_COOKIE['unific_newsletter_subscription'] ) ) : 'false';
			setcookie( 'unific_newsletter_subscription', '', time() - ( 60 * 60 ), '/' );
			if ( isset( $unific_newsletter_subscription ) ) {
				update_post_meta( $order_id, 'unific_newsletter_subscription', $unific_newsletter_subscription );
				// save user meta.
				unific_checkout_save_user_meta( $order_id, 'unific_newsletter_subscription', $unific_newsletter_subscription );
			}
		}
	}

	if ( ! function_exists( 'unific_checkout_save_user_meta' ) ) {
		/**
		 * Unific_checkout_save_user_meta
		 *
		 * @param int $order_id    The ID of the order being processed.
		 * @param int $meta_key    The meta key.
		 * @param int $meta_value  The meta value.
		 */
		function unific_checkout_save_user_meta( $order_id, $meta_key, $meta_value ) {
			$order = wc_get_order( $order_id );
			if ( ! empty( $order ) ) {
				$user_id = $order->get_user_id();
				if ( ! empty( $user_id ) ) {
					update_user_meta( $user_id, $meta_key, $meta_value );
				}
			}
		}
	}

	/* ***************************** Newly added for WC abandoned cart  ************************************ */
	if ( ! function_exists( 'unific_checkout_create' ) ) {
		/**
		 * Unific_cart_hash value
		 *
		 * @param int $unific_cart_hash value.
		 */
		function unific_checkout_create( $unific_cart_hash ) {
			// add code here.
			// return WC_API_Unific_Checkout::get_checkout($unific_cart_hash).

			return $unific_cart_hash;
		}
	}

	if ( ! function_exists( 'unific_checkout_update' ) ) {
		/**
		 * Unific_cart_hash value
		 *
		 * @param int $unific_cart_hash value.
		 */
		function unific_checkout_update( $unific_cart_hash ) {
			// add code here.
			// return WC_API_Unific_Checkout::get_checkout($unific_cart_hash).
			// return array("checkout" => $unific_cart_hash).

			return $unific_cart_hash;
		}
	}

	if ( ! function_exists( 'unific_plugin_uninstall' ) ) {
		/**
		 * Unific_plugin_data value
		 *
		 * @param int $unific_plugin_data value.
		 */
		function unific_plugin_uninstall( $unific_plugin_data ) {
			return $unific_plugin_data;
		}
	}

	if ( ! function_exists( 'load_user_data_js' ) ) {
		/**
		 * Load_user_data_js
		 */
		function load_user_data_js() {
			$unific_cart_hash   = ! empty( $_COOKIE['unific_cart_hash'] ) ? sanitize_text_field( wp_unslash( $_COOKIE['unific_cart_hash'] ) ) : null;
			$email              = null;
			$billing_first_name = null;
			$billing_last_name  = null;

			if ( ! empty( $unific_cart_hash ) ) {
				global $wpdb;

				$unific_table_name = $wpdb->prefix . 'usermeta_unific';
				// phpcs:ignore
				$results = $wpdb->get_results( $wpdb->prepare( 'SELECT meta_key,meta_value FROM %1s WHERE entity_id = %d && entity_type = %s && (meta_key = %s || meta_key = %s || meta_key = %s)', $unific_table_name, $unific_cart_hash, 'checkout', 'email', 'first_name', 'last_name' ) );

				if ( ! empty( $results ) ) {
					foreach ( $results as $result ) {
						if ( ! empty( $result->meta_key ) && ! empty( $result->meta_value ) ) {
							switch ( $result->meta_key ) {
								case 'email':
									$email = $result->meta_value;
									break;
								case 'first_name':
									$billing_first_name = stripslashes( $result->meta_value );
									break;
								case 'last_name':
									$billing_last_name = stripslashes( $result->meta_value );
									break;
							}
						}
					}
				} elseif ( is_user_logged_in() ) {
					$current_user       = wp_get_current_user();
					$email              = ( ! empty( $current_user->user_email ) ) ? $current_user->user_email : null;
					$billing_first_name = ( ! empty( $current_user->user_firstname ) ) ? $current_user->user_firstname : null;
					$billing_last_name  = ( ! empty( $current_user->user_lastname ) ) ? $current_user->user_lastname : null;
				}
			}

			// phpcs:ignore WordPress.WP.EnqueuedResourceParameters.NoExplicitVersion
			wp_register_script( 'unific', plugin_dir_url( __FILE__ ) . 'public/js/unific.js', array( 'jquery' ), false, false );
			wp_enqueue_script( 'unific' );
			wp_localize_script(
				'unific',
				'unific_object',
				array(
					'nonce'              => wp_create_nonce( 'ahsdsad66a7sas7' ),
					'email'              => $email,
					'billing_first_name' => $billing_first_name,
					'billing_last_name'  => $billing_last_name,
					'admin_ajax_url'     => get_admin_url() . 'admin-ajax.php',
				)
			);
		}
	}

	if ( ! function_exists( 'starts_with' ) ) {
		/**
		 * Startswith
		 *
		 * @param int $haystack value.
		 * @param int $needle value.
		 */
		function starts_with( $haystack, $needle ) {
			$length = strlen( $needle );
			return ( substr( $haystack, 0, $length ) === $needle );
		}
	}

	if ( ! function_exists( 'save_user_callback' ) ) {
		/**
		 * Save_user_callback
		 */
		function save_user_callback() {
			global $woocommerce;

			$cart             = array();
			$billing_address  = array();
			$shipping_address = array();
			if ( ! empty( $_POST['billing_email'] ) ) { // phpcs:ignore
				$email = strtolower( sanitize_text_field( wp_unslash( $_POST['billing_email'] ) ) ); // phpcs:ignore
			} else {
				$email = null;
			}
			if ( ! empty( $_POST['billing_first_name'] ) ) { // phpcs:ignore
				$billing_first_name = sanitize_text_field( wp_unslash( $_POST['billing_first_name'] ) ); // phpcs:ignore
			} else {
				$billing_first_name = '';
			}
			if ( ! empty( $_POST['billing_last_name'] ) ) { // phpcs:ignore
				$billing_last_name = sanitize_text_field( wp_unslash( $_POST['billing_last_name'] ) ); // phpcs:ignore
			} else {
				$billing_last_name = '';
			}
			if ( function_exists( 'WC' ) ) {
				$cart['cart'] = WC()->session->cart;
			} else {
				$cart['cart'] = $woocommerce->session->cart;
			}

			if ( ! empty( $_POST['unific_newsletter_subscription'] ) ) { // phpcs:ignore
				$unific_newsletter_subscription = 'true';
			} else {
				$unific_newsletter_subscription = 'false';
			}
			setcookie( 'unific_newsletter_subscription', $unific_newsletter_subscription, time() + ( 60 * 60 ), '/' );

			foreach ( $_POST as $key => $val ) { // phpcs:ignore
				if ( starts_with( $key, 'billing_' ) ) {
					$billing_address[ str_replace( 'billing_', '', $key ) ] = $val;
				}
				if ( starts_with( $key, 'shipping_' ) ) {
					$shipping_address[ str_replace( 'shipping_', '', $key ) ] = $val;
				}
			}

			// If a record is present in the users table for the same email id.
			if ( ! empty( $email ) && ! empty( $billing_first_name ) && ! empty( $billing_last_name ) && ! empty( $cart['cart'] ) ) {
				$unific_cart_hash = ! empty( $_COOKIE['unific_cart_hash'] ) ? sanitize_text_field( wp_unslash( $_COOKIE['unific_cart_hash'] ) ) : null;
				if ( empty( $unific_cart_hash ) ) {
					$unific_cart_hash = (int) ( microtime( true ) * 1000 );
					setcookie( 'unific_cart_hash', $unific_cart_hash, time() + ( 60 * 60 ), '/' );
				}
				if ( ! empty( $unific_cart_hash ) ) {
					save_checkout_data( $unific_cart_hash, $email, $billing_first_name, $billing_last_name, $billing_address, $shipping_address, $cart['cart'] );
				}
			}
			// }
		}
	}

	if ( ! function_exists( 'atleast_wc_version' ) ) {
		/**
		 * WC version
		 *
		 * @param boolean $version value.
		 */
		function atleast_wc_version( $version = '3.5.0' ) {
			if ( class_exists( 'WooCommerce' ) ) {
				global $woocommerce;
				if ( version_compare( $woocommerce->version, $version, '>=' ) ) {
					return true;
				}
			}
			return false;
		}
	}

	if ( ! function_exists( 'save_checkout_data' ) ) {
		/**
		 * SaveCheckoutData
		 *
		 * @param int    $unific_cart_hash       Value.
		 * @param string $email                  Value.
		 * @param string $billing_first_name     Value.
		 * @param string $billing_last_name      Value.
		 * @param string $billing_address        Value.
		 * @param string $shipping_address       Value.
		 * @param string $cart                   Value.
		 * @param int    $user_id                Value.
		 */
		function save_checkout_data( $unific_cart_hash, $email, $billing_first_name, $billing_last_name, $billing_address, $shipping_address, $cart, $user_id = 0 ) {
			global $wpdb;

			if ( ! empty( $email ) && ! empty( $billing_first_name ) && ! empty( $billing_last_name ) && ! empty( $cart ) && ! empty( $unific_cart_hash ) ) {
				$cart_data         = serialize( $cart ); // phpcs:ignore
				$unific_table_name = $wpdb->prefix . 'usermeta_unific';
				// phpcs:ignore
				$results                     = $wpdb->get_results( $wpdb->prepare( 'SELECT unific_meta_id,meta_key,meta_value FROM %1s WHERE entity_id = %d && entity_type = %s', $unific_table_name, $unific_cart_hash, 'checkout' ) );
				$checkout_data_already_exist = false;
				if ( ! empty( $results ) ) {
					$checkout_data_already_exist = true;
					foreach ( $results as $result ) {
						if ( ! empty( $result->meta_key ) && ! empty( $result->meta_value ) ) {
							switch ( $result->meta_key ) {
								case 'email':
									$email_result = $result;
									break;
								case 'first_name':
									$first_name_result = $result;
									break;
								case 'last_name':
									$last_name_result = $result;
									break;
								case 'user_id':
									$user_id_result = $result;
									break;
								case 'billing_address':
									$billing_address_result = $result;
									break;
								case 'shipping_address':
									$shipping_address_result = $result;
									break;
								case 'unific_persistent_cart':
									$cart_result = $result;
									break;
							}
						}
					}
				}

				$current_time = current_time( 'mysql' );

				$is_update         = false;
				$unific_table_name = $wpdb->prefix . 'usermeta_unific';

				$first_name_query       = '';
				$last_name_query        = '';
				$cart_query             = '';
				$billing_address_query  = '';
				$shipping_address_query = '';
				if ( ! empty( $user_id ) ) {
					if ( ! empty( $user_id_result ) && ! empty( $user_id_result->unific_meta_id ) ) {
						if ( $user_id_result->meta_value !== $user_id ) {
							// phpcs:ignore
							$wpdb->query( $wpdb->prepare( 'UPDATE %1s SET meta_value = %s, modified = %s WHERE unific_meta_id = %d', $unific_table_name, $user_id, $current_time, $user_id_result->unific_meta_id ) );
						}
					} else {
						// phpcs:ignore
						$wpdb->query( $wpdb->prepare( 'INSERT INTO %1s( entity_id, entity_type, meta_key, meta_value, created, modified) VALUES ( %s, %s, %s, %s, %s, %s )', $unific_table_name, $unific_cart_hash, 'checkout', 'user_id', $user_id, $current_time, $current_time ) );
					}
				}

				if ( ! empty( $email_result ) && ! empty( $email_result->unific_meta_id ) ) {
					if ( $email_result->meta_value !== $email ) {
						// phpcs:ignore
						$wpdb->query( $wpdb->prepare( 'UPDATE %1s SET meta_value = %s, modified = %s WHERE unific_meta_id = %d ', $unific_table_name, $email, $current_time, $email_result->unific_meta_id ) );
						$is_update = true;
					}
				} else {
					// phpcs:ignore
					$wpdb->query( $wpdb->prepare( 'INSERT INTO %1s( entity_id, entity_type, meta_key, meta_value, created, modified) VALUES ( %s, %s, %s, %s, %s, %s )', $unific_table_name, $unific_cart_hash, 'checkout', 'email', $email, $current_time, $current_time ) );
				}

				if ( ! empty( $first_name_result ) && ! empty( $first_name_result->unific_meta_id ) ) {
					if ( $first_name_result->meta_value !== $billing_first_name ) {
						// phpcs:ignore
						$wpdb->query( $wpdb->prepare( 'UPDATE %1s SET meta_value = %s, modified = %s WHERE unific_meta_id = %d ', $unific_table_name, $wpdb->_real_escape( $billing_first_name ), $current_time, $first_name_result->unific_meta_id ) );
						$is_update = true;
					}
				} else {
					// phpcs:ignore
					$wpdb->query( $wpdb->prepare( 'INSERT INTO %1s( entity_id, entity_type, meta_key, meta_value, created, modified) VALUES ( %s, %s, %s, %s, %s, %s )', $unific_table_name, $unific_cart_hash, 'checkout', 'first_name', $wpdb->_real_escape( $billing_first_name ), $current_time, $current_time ) );
				}

				if ( ! empty( $last_name_result ) && ! empty( $last_name_result->unific_meta_id ) ) {
					if ( $last_name_result->meta_value !== $billing_last_name ) {
						// phpcs:ignore
						$wpdb->query( $wpdb->prepare( 'UPDATE %1s SET meta_value = %s, modified = %s WHERE unific_meta_id = %d ', $unific_table_name, $wpdb->_real_escape( $billing_last_name ), $current_time, $last_name_result->unific_meta_id ) );
						$is_update = true;
					}
				} else {
					// phpcs:ignore
					$wpdb->query( $wpdb->prepare( 'INSERT INTO %1s( entity_id, entity_type, meta_key, meta_value, created, modified) VALUES ( %s, %s, %s, %s, %s, %s )', $unific_table_name, $unific_cart_hash, 'checkout', 'last_name', $wpdb->_real_escape( $billing_last_name ), $current_time, $current_time ) );
				}

				if ( ! empty( $cart_result ) && ! empty( $cart_result->unific_meta_id ) ) {
					if ( $cart_result->meta_value !== $cart_data ) {
						// phpcs:ignore
						$wpdb->query( $wpdb->prepare( 'UPDATE %1s SET meta_value = %s, modified = %s WHERE unific_meta_id = %d ', $unific_table_name, $wpdb->_real_escape( $cart_data ), $current_time, $cart_result->unific_meta_id ) );
						$is_update = true;
					}
				} else {
					// phpcs:ignore
					$wpdb->query( $wpdb->prepare( 'INSERT INTO %1s( entity_id, entity_type, meta_key, meta_value, created, modified) VALUES ( %s, %s, %s, %s, %s, %s )', $unific_table_name, $unific_cart_hash, 'checkout', 'unific_persistent_cart', $wpdb->_real_escape( $cart_data ), $current_time, $current_time ) );
				}

				if ( ! empty( $billing_address_result ) && ! empty( $billing_address_result->unific_meta_id ) ) {
					if ( serialize( $billing_address ) !== $billing_address_result->meta_value ) { // phpcs:ignore
						// phpcs:ignore
						$wpdb->query( $wpdb->prepare( 'UPDATE %1s SET meta_value = %s, modified = %s WHERE unific_meta_id = %d ', $unific_table_name, $wpdb->_real_escape( serialize( $billing_address ) ), $current_time, $billing_address_result->unific_meta_id ) );
						$is_update = true;
					}
				} else {
					// phpcs:ignore
					$wpdb->query( $wpdb->prepare( 'INSERT INTO %1s( entity_id, entity_type, meta_key, meta_value, created, modified) VALUES ( %s, %s, %s, %s, %s, %s )', $unific_table_name, $unific_cart_hash, 'checkout', 'billing_address', $wpdb->_real_escape( serialize( $billing_address ) ), $current_time, $current_time ) );
				}

				if ( ! empty( $shipping_address_result ) && ! empty( $shipping_address_result->unific_meta_id ) ) {
					if ( serialize( $shipping_address ) !== $shipping_address_result->meta_value ) { // phpcs:ignore
						// phpcs:ignore
						$wpdb->query( $wpdb->prepare( 'UPDATE %1s SET meta_value = %s, modified = %s WHERE unific_meta_id = %d ', $unific_table_name, $wpdb->_real_escape( serialize( $shipping_address ) ), $current_time, $shipping_address_result->unific_meta_id ) );
						$is_update = true;
					}
				} else {
					// phpcs:ignore
					$wpdb->query( $wpdb->prepare( 'INSERT INTO %1s( entity_id, entity_type, meta_key, meta_value, created, modified) VALUES ( %s, %s, %s, %s, %s, %s )', $unific_table_name, $unific_cart_hash, 'checkout', 'shipping_address', $wpdb->_real_escape( serialize( $shipping_address ) ), $current_time, $current_time ) );
				}

				$unific_cart_data = unific_get_checkout( $unific_cart_hash );

				// send customer update webhook for logged in user for checkout create.
				if ( ( ! $is_update ) && ! empty( $user_id ) && ( ! $checkout_data_already_exist ) ) {
					do_action( 'woocommerce_update_customer', $user_id );
				}

				if ( atleast_wc_version( '3.5.0' ) ) {
					if ( $is_update ) {
						// fire below action to checkout update webhook.
						do_action( 'woocommerce_unific_checkout_update_action', $unific_cart_data );
					} elseif ( ! $checkout_data_already_exist ) {
						// fire below action to checkout create webhook.
						do_action( 'woocommerce_unific_checkout_create_action', $unific_cart_data );
					}
				} elseif ( $is_update ) {
						// fire below action to checkout update webhook.
						do_action( 'woocommerce_unific_checkout_update_action', wp_json_encode( $unific_cart_data ) );
				} elseif ( ! $checkout_data_already_exist ) {
					// fire below action to checkout create webhook.
					do_action( 'woocommerce_unific_checkout_create_action', wp_json_encode( $unific_cart_data ) );
				}
			}
		}
	}

	if ( ! function_exists( 'get_latest_date' ) ) {
		/**
		 * Get_latest_date
		 *
		 * @param date $current time.
		 * @param date $from_source time.
		 */
		function get_latest_date( $current, $from_source ) {
			if ( ! empty( $from_source ) ) {
				if ( empty( $current ) ) {
					$current = $from_source;
				} elseif ( $from_source > $current ) {
						$current = $from_source;
				}
			}
			return $current;
		}
	}

	if ( ! function_exists( 'unific_get_checkout' ) ) {
		/**
		 * Unific_get_checkout
		 *
		 * @param int $id value.
		 */
		function unific_get_checkout( $id ) {
			global $wpdb;
			global $woocommerce;
			$cart_url = wc_get_cart_url();

			$ret               = array();
			$customer_data     = array();
			$created           = null;
			$modified          = null;
			$cart_url          = $cart_url . '?unific_cart_hash=' . $id;
			$unific_table_name = $wpdb->prefix . 'usermeta_unific';

			// phpcs:ignore
			$results = $wpdb->get_results( $wpdb->prepare( 'SELECT meta_key,meta_value,created,modified FROM %1s WHERE entity_id = %d && entity_type = %s', $unific_table_name, $id, 'checkout' ) );

			if ( ! empty( $results ) ) {
				foreach ( $results as $result ) {
					if ( ! empty( $result->meta_key ) && ! empty( $result->meta_value ) ) {
						switch ( $result->meta_key ) {
							case 'email':
								$customer_data['email'] = $result->meta_value;
								if ( ! empty( $result->created ) ) {
									$created = get_latest_date( $created, $result->created );
								}
								if ( ! empty( $result->modified ) ) {
									$modified = get_latest_date( $created, $result->modified );
								}
								break;
							case 'first_name':
								$customer_data['first_name'] = stripslashes( $result->meta_value );
								if ( ! empty( $result->created ) ) {
									$created = get_latest_date( $created, $result->created );
								}
								if ( ! empty( $result->modified ) ) {
									$modified = get_latest_date( $created, $result->modified );
								}
								break;
							case 'last_name':
								$customer_data['last_name'] = stripslashes( $result->meta_value );
								if ( ! empty( $result->created ) ) {
									$created = get_latest_date( $created, $result->created );
								}
								if ( ! empty( $result->modified ) ) {
									$modified = get_latest_date( $created, $result->modified );
								}
								break;
							case 'user_id':
								$customer_data['id'] = $result->meta_value;
								break;
							case 'billing_address':
								$customer_data['billing_address'] = ( ! empty( $result->meta_value ) ) ? custom_unserialize( stripslashes( $result->meta_value ) ) : array();
								break;
							case 'shipping_address':
								$customer_data['shipping_address'] = ( ! empty( $result->meta_value ) ) ? custom_unserialize( stripslashes( $result->meta_value ) ) : array();
								if ( ! empty( $customer_data['shipping_address']['method'] ) ) {
									unset( $customer_data['shipping_address']['method'] );
								}
								break;
							case 'unific_persistent_cart':
								$cart = custom_unserialize( stripslashes( $result->meta_value ) );
								if ( ! empty( $cart ) ) {
									$ret['id']       = $id;
									$ret['cart_url'] = $cart_url;
									$ret['cart']     = $cart;
								}
								if ( ! empty( $result->created ) ) {
									$created = get_latest_date( $created, $result->created );
								}
								if ( ! empty( $result->modified ) ) {
									$modified = get_latest_date( $created, $result->modified );
								}
								break;
						}
					}
				}
			}

			if ( ! empty( $ret['cart'] ) ) {
				if ( ! empty( $created ) ) {
					$ret['created_at'] = convert_to_datetime( $created );
				}
				if ( ! empty( $modified ) ) {
					$ret['updated_at'] = convert_to_datetime( $modified );
				}
				if ( ! empty( $customer_data ) ) {
					$ret['customer'] = $customer_data;
				}
			}
			return $ret;
		}
	}

	if ( ! function_exists( 'convert_to_datetime' ) ) {
		/**
		 * Convert_to_datetime value
		 *
		 * @param date $timestamp value.
		 * @param date $convert_to_utc value.
		 * @param date $convert_to_gmt value.
		 */
		function convert_to_datetime( $timestamp, $convert_to_utc = false, $convert_to_gmt = false ) {
			if ( $convert_to_gmt ) {
				if ( is_numeric( $timestamp ) ) {
					$timestamp = gmdate( 'Y-m-d H:i:s', $timestamp );
				}

				$timestamp = get_gmt_from_date( $timestamp );
			}

			if ( $convert_to_utc ) {
				$timezone = new DateTimeZone( wc_timezone_string() );
			} else {
				$timezone = new DateTimeZone( 'UTC' );
			}

			try {

				if ( is_numeric( $timestamp ) ) {
					$date = new DateTime( "@{$timestamp}" );
				} else {
					$date = new DateTime( $timestamp, $timezone );
				}

				// convert to UTC by adjusting the time based on the offset of the site's timezone.
				if ( $convert_to_utc ) {
					$date->modify( -1 * $date->getOffset() . ' seconds' );
				}
			} catch ( Exception $e ) {

				$date = new DateTime( '@0' );
			}

			return $date->format( 'Y-m-d\TH:i:s\Z' );
		}
	}

	if ( ! function_exists( 'unific_save_custom_persistant_cart' ) ) {
		/**
		 * Unific_save_custom_persistant_cart value
		 */
		function unific_save_custom_persistant_cart() {
			global $wpdb;
			global $woocommerce;
			$user_id      = get_current_user_id();
			$current_cart = null;
			if ( function_exists( 'WC' ) && ! empty( WC()->session ) ) {
				if ( ! empty( WC()->session->cart ) ) {
					$current_cart = WC()->session->cart;
				} else {
					$current_cart = WC()->session->get( 'cart' );
				}
			} elseif ( ! empty( $woocommerce ) && ! empty( $woocommerce->session ) && ! empty( $woocommerce->session->cart ) ) {
				$current_cart = $woocommerce->session->cart;
			}
			$unific_cart_hash = ! empty( $_COOKIE['unific_cart_hash'] ) ? sanitize_text_field( wp_unslash( $_COOKIE['unific_cart_hash'] ) ) : null;
			if ( ! empty( $user_id ) && ! empty( $current_cart ) && is_array( $current_cart ) && count( $current_cart ) > 0 ) {
				if ( empty( $unific_cart_hash ) ) {
					$unific_cart_hash = (int) ( microtime( true ) * 1000 );
					setcookie( 'unific_cart_hash', $unific_cart_hash, time() + ( 60 * 60 ), '/' );
				}

				$unific_table_name = $wpdb->prefix . 'usermeta_unific';
				// phpcs:ignore
				$results = $wpdb->get_results( $wpdb->prepare( 'SELECT unific_meta_id,meta_key,meta_value FROM %1s WHERE entity_id = %d && entity_type = %s', $unific_table_name, $unific_cart_hash, 'checkout' ) );
				if ( ! empty( $results ) ) {
					foreach ( $results as $result ) {
						if ( ! empty( $result->meta_key ) && ! empty( $result->meta_value ) ) {
							switch ( $result->meta_key ) {
								case 'email':
									$email_result = $result;
									break;
								case 'first_name':
									$first_name_result = $result;
									break;
								case 'last_name':
									$last_name_result = $result;
									break;
								case 'billing_address':
									$billing_address_result = $result;
									break;
								case 'shipping_address':
									$shipping_address_result = $result;
									break;
							}
						}
					}
				}

				$current_user                  = wp_get_current_user();
				$current_user_billing_address  = array();
				$current_user_shipping_address = array();
				if ( ! empty( $user_id ) ) {
					$current_user_meta = get_user_meta( $user_id, '', true );
					if ( ! empty( $current_user_meta ) ) {
						foreach ( $current_user_meta as $key => $val ) {
							if ( starts_with( $key, 'billing_' ) ) {
								if ( ! empty( $val['0'] ) ) {
									$current_user_billing_address[ str_replace( 'billing_', '', $key ) ] = $val['0'];
								} else {
									$current_user_billing_address[ str_replace( 'billing_', '', $key ) ] = '';
								}
							}
							if ( starts_with( $key, 'shipping_' ) ) {
								if ( ! empty( $val['0'] ) ) {
									$current_user_shipping_address[ str_replace( 'shipping_', '', $key ) ] = $val['0'];
								} else {
									$current_user_shipping_address[ str_replace( 'shipping_', '', $key ) ] = '';
								}
							}
						}
					}
				}

				if ( ! empty( $email_result ) && ! empty( $email_result->meta_value ) ) {
					$email = $email_result->meta_value;
				} else {
					$email = $current_user->user_email;
				}
				if ( ! empty( $first_name_result ) && ! empty( $first_name_result->meta_value ) ) {
					$billing_first_name = $first_name_result->meta_value;
				} else {
					$billing_first_name = ( ! empty( $current_user->user_firstname ) ) ? $current_user->user_firstname : null;
				}
				if ( ! empty( $last_name_result ) && ! empty( $last_name_result->meta_value ) ) {
					$billing_last_name = $last_name_result->meta_value;
				} else {
					$billing_last_name = ( ! empty( $current_user->user_lastname ) ) ? $current_user->user_lastname : null;
				}

				if ( ! empty( $billing_address_result ) && ! empty( $billing_address_result->meta_value ) ) {
					$billing_address = custom_unserialize( $billing_address_result->meta_value );
					if ( empty( $billing_address ) ) {
						$billing_address = $current_user_billing_address;
					}
				} else {
					$billing_address = $current_user_billing_address;
				}

				if ( ! empty( $shipping_address_result ) && ! empty( $shipping_address_result->meta_value ) ) {
					$shipping_address = custom_unserialize( $shipping_address_result->meta_value );
					if ( empty( $shipping_address ) ) {
						$shipping_address = $current_user_shipping_address;
					}
				} else {
					$shipping_address = $current_user_shipping_address;
				}

				if ( ! empty( $unific_cart_hash ) && ! empty( $email ) && ! empty( $billing_first_name ) && ! empty( $billing_last_name ) ) {
					save_checkout_data( $unific_cart_hash, $email, $billing_first_name, $billing_last_name, $billing_address, $shipping_address, $current_cart, $user_id );
				}
			}
		}
	}

	if ( ! function_exists( 'custom_unserialize' ) ) {
		/**
		 * Custom_unserialize
		 *
		 * @param string $meta_value value.
		 */
		function custom_unserialize( $meta_value ) {
			if ( @unserialize( $meta_value ) !== true && preg_match( '/^[aOs]:/', $meta_value ) ) { // phpcs:ignore
				$meta_value = preg_replace_callback(
					'/s\:(\d+)\:\"(.*?)\";/s',
					function ( $matches ) {
						return 's:' . strlen( $matches[2] ) . ':"' . $matches[2] . '";';
					},
					$meta_value
				);
			}
			return unserialize( $meta_value ); // phpcs:ignore
		}
	}

	if ( ! function_exists( 'check_and_load_cart_from_abandoned_url' ) ) {
		/**
		 * Check_and_load_cart_from_abandoned_url value
		 */
		function check_and_load_cart_from_abandoned_url() {
			if ( is_cart() && ! empty( $_GET['unific_cart_hash'] ) ) {  // phpcs:ignore
				global $wpdb, $woocommerce;

				$unific_cart_hash = sanitize_text_field( wp_unslash( $_GET['unific_cart_hash'] ) ); // phpcs:ignore
				$cart_url         = wc_get_cart_url();

				$unific_table_name = $wpdb->prefix . 'usermeta_unific';
        		// phpcs:ignore
				$meta = $wpdb->get_row( $wpdb->prepare( 'SELECT meta_value FROM %1s WHERE entity_id = %d && entity_type = %s && meta_key = %s', $unific_table_name, $unific_cart_hash, 'checkout', 'unific_persistent_cart' ) );
				unset( $cart_query );

				if ( ! empty( $meta->meta_value ) ) {
					$meta_value = custom_unserialize( $meta->meta_value );
					if ( ! empty( $meta_value ) ) {
						// add products of meta values in cart.
						foreach ( $meta_value as $line_item ) {
							if ( ! empty( $line_item['product_id'] ) && ! empty( $line_item['quantity'] ) ) {
								// added this code to show variants by Vilas.
								$variation_id = 0;
								$variation    = array();
								if ( ! empty( $line_item ['variation_id'] ) ) {
									$variation_id = $line_item['variation_id'];
								}
								if ( ! empty( $line_item['variation'] ) ) {
									$variation = $line_item['variation'];
								}
								WC()->cart->add_to_cart( $line_item['product_id'], $line_item['quantity'], $variation_id, $variation );
							}
						}
						setcookie( 'unific_cart_hash', $unific_cart_hash, time() + ( 60 * 60 ), '/' );
						wp_safe_redirect( $cart_url );
					}
				}
			}
		}
	}

	if ( ! function_exists( 'logout_callback' ) ) {
		/**
		 * Logout callback
		 */
		function logout_callback() {
			setcookie( 'unific_cart_hash', '', time() - ( 60 * 60 ), '/' );
		}
	}

	if ( ! function_exists( 'add_meta_to_woocommerce_webhook_payload' ) ) {
		/**
		 * Add Meta to WC
		 *
		 * @param int    $payload      Value.
		 * @param string $resource_name     Value.
		 * @param int    $resource_id  Value.
		 */
		function add_meta_to_woocommerce_webhook_payload( $payload, $resource_name, $resource_id ) {
			if ( 'order' === $resource_name ) {
				if ( ! empty( $resource_id ) ) {
					$order_meta = get_post_meta( $resource_id, '', true );
					$meta       = array();
					if ( ! empty( $order_meta ) ) {
						foreach ( $order_meta as $key => $val ) {
							if ( ! empty( $key ) && ! empty( $val ) ) {
								// exclude default meta keys which starts with underscore.
								if ( ! ( strpos( $key, '_' ) === 0 ) ) {
									$meta[ $key ] = $val[0];
								}
							}
						}
					}
					if ( ! empty( $payload['order'] ) ) {
						$payload['order']['order_meta'] = $meta;
					} elseif ( ! empty( $payload ) ) {
						$payload['order_meta'] = $meta;
					}
					return $payload;
				}
			}
			return $payload;
		}
	}

	// fire customer update webhook for order create/update.
	if ( ! function_exists( 'action_woocommerce_order' ) ) {
		/**
		 * Action_woocommerce_order value
		 *
		 * @param int $order_id value.
		 */
		function action_woocommerce_order( $order_id ) {
			$order   = wc_get_order( $order_id );
			$user_id = $order->get_user_id();
			if ( ! empty( $user_id ) ) {
				do_action( 'woocommerce_update_customer', $user_id );
			}
		}
	}

	/* ***************************** End here  ************************************ */

	if ( ! function_exists( 'unific_parse_request' ) ) {
		/**
		 * Unific_parse_request value
		 *
		 * @param string $wp value.
		 */
		function unific_parse_request( $wp ) {
			$query_params = $wp->query_vars;
			if ( ! empty( $query_params['affiliate'] ) ) {
				setcookie( 'unific_affiliate_id', trim( $query_params['affiliate'] ), time() + ( 3600 * 24 * 30 ), '/' );
			}
		}
	}

	if ( ! function_exists( 'unific_add_query_vars_filter' ) ) {
		/**
		 * Unific_add_query_vars_filter value
		 *
		 * @param int $vars value.
		 */
		function unific_add_query_vars_filter( $vars ) {
			$vars[] = 'affiliate';
			return $vars;
		}
	}

	/**
	 * Add WooCommerce additional Checkbox checkout field for newsletter subscription
	 */
	function add_newsletter_subscription_checkbox() {
		$newsletter_enabled = get_option( 'enable_newsletter', 'yes' );

		if ( 'yes' === $newsletter_enabled ) {
			woocommerce_form_field(
				'unific_newsletter_subscription',
				array(
					'type'        => 'checkbox',
					'class'       => array( 'form-row', 'custom-checkbox-class' ), // Add your custom class here.
					'label_class' => array( 'woocommerce-form__label woocommerce-form__label-for-checkbox checkbox' ),
					'input_class' => array( 'woocommerce-form__input woocommerce-form__input-checkbox input-checkbox' ),
					'label'       => __( 'Email me with news and offers' ),
				),
				WC()->session->get( 'unific_newsletter_subscription' )
			);
		}
	}

	/**
	 * Save newsletter_subscription_checkbox_value
	 */
	function save_newsletter_subscription_checkbox_value() {
		$subscription_value = isset( $_POST['unific_newsletter_subscription'] ) ? true : false; // phpcs:ignore
		WC()->session->set( 'unific_newsletter_subscription', $subscription_value );
	}

	if ( is_admin() ) {
		require_once __DIR__ . '/admin/unific-admin.php';
	}

	add_action( 'woocommerce_review_order_before_submit', 'add_newsletter_subscription_checkbox', 10 );
	add_action( 'woocommerce_checkout_update_order_review', 'save_newsletter_subscription_checkbox_value' );

	add_filter( 'query_vars', 'unific_add_query_vars_filter' );
	add_action( 'parse_query', 'unific_parse_request' );
	add_action( 'woocommerce_checkout_update_order_meta', 'unific_save_extra_checkout_fields', 10, 1 );

	/* ***************************** Newly added for WC abandoned cart  ************************************ */
	add_filter( 'woocommerce_webhook_payload', 'add_meta_to_woocommerce_webhook_payload', 10, 4 );

	add_action( 'wp_ajax_save_user', 'save_user_callback' );
	add_action( 'wp_ajax_nopriv_save_user', 'save_user_callback' );

	add_action( 'woocommerce_after_checkout_billing_form', 'load_user_data_js' );

	add_action( 'template_redirect', 'check_and_load_cart_from_abandoned_url', 10, 1 );
	add_action( 'woocommerce_checkout_init', 'unific_save_custom_persistant_cart', 10, 1 );
	add_action( 'wp_logout', 'logout_callback' );

	add_action( 'woocommerce_new_order', 'action_woocommerce_order', 10, 1 );
	add_action( 'woocommerce_update_order', 'action_woocommerce_order', 10, 1 );

	/* ******************** Custom hook function *********************** */
	add_action( 'woocommerce_unific_checkout_update_action', 'unific_checkout_update', 10, 1 );
	add_action( 'woocommerce_unific_checkout_create_action', 'unific_checkout_create', 10, 1 );
	add_action( 'woocommerce_unific_uninstall_action', 'unific_plugin_uninstall', 10, 1 );

	/* ********************* Custom hook function end ********************* */

	add_action(
		'woocommerce_api_loaded',
		function () {
			include_once 'includes/api/class-wc-api-unific-checkout.php';
		}
	);

	add_filter(
		'woocommerce_api_classes',
		function ( $classes ) {
			$classes[] = 'WC_API_Unific_Checkout';
			return $classes;
		}
	);

	/**
	 * Hubspotutk token.
	 */
	function hubspotutk_submission_script() {
		wp_enqueue_script( 'hubspotutk-submission-script', plugin_dir_url( __FILE__ ) . 'public/js/hubspotutk-submission-script.js', array( 'jquery' ), '1.0', true );
	}

	add_action( 'wp_enqueue_scripts', 'hubspotutk_submission_script' );

	/* ***************************** End here  ************************************ */
}
